<?php
/**
 * রমযান বাড়ি — PHP API Backend
 * File: api.php
 * 
 * এই file টি আপনার hosting এ upload করুন।
 * config.php তে database credentials দিন।
 */

// ============================================================
// CONFIG — এখানে আপনার cPanel database info দিন
// ============================================================
define('DB_HOST', 'localhost');
define('DB_NAME', 'ramadan_db');      // আপনার database নাম
define('DB_USER', 'your_db_user');    // আপনার database username
define('DB_PASS', 'your_db_password');// আপনার database password
define('ADMIN_SESSION_KEY', 'rb_admin_2026');

// ============================================================
// CORS & Headers
// ============================================================
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Admin-Token');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ============================================================
// Database Connection
// ============================================================
function getDB() {
    static $pdo = null;
    if ($pdo === null) {
        try {
            $pdo = new PDO(
                'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4',
                DB_USER, DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::MYSQL_ATTR_INIT_COMMAND => 'SET NAMES utf8mb4'
                ]
            );
        } catch (PDOException $e) {
            jsonError('Database connection failed: ' . $e->getMessage(), 500);
        }
    }
    return $pdo;
}

// ============================================================
// Helper Functions
// ============================================================
function jsonResponse($data, $code = 200) {
    http_response_code($code);
    echo json_encode(['success' => true, 'data' => $data], JSON_UNESCAPED_UNICODE);
    exit;
}

function jsonError($msg, $code = 400) {
    http_response_code($code);
    echo json_encode(['success' => false, 'error' => $msg], JSON_UNESCAPED_UNICODE);
    exit;
}

function getInput() {
    $raw = file_get_contents('php://input');
    return json_decode($raw, true) ?: [];
}

function isAdmin() {
    $token = $_SERVER['HTTP_X_ADMIN_TOKEN'] ?? '';
    if (empty($token)) return false;
    // Token = base64(username:timestamp:hash)
    $decoded = base64_decode($token);
    if (!$decoded) return false;
    list($user, $ts, $hash) = explode(':', $decoded, 3) + ['', 0, ''];
    // Token valid for 24 hours
    if (time() - (int)$ts > 86400) return false;
    $expected = hash('sha256', $user . $ts . ADMIN_SESSION_KEY);
    return hash_equals($expected, $hash);
}

function requireAdmin() {
    if (!isAdmin()) jsonError('Unauthorized — Admin access required', 401);
}

// ============================================================
// Router
// ============================================================
$action = $_GET['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

switch ($action) {
    // --- Public endpoints ---
    case 'get_all':       handleGetAll(); break;
    case 'get_prayers':   handleGetPrayers(); break;
    case 'get_schedule':  handleGetSchedule(); break;
    case 'get_duas':      handleGetDuas(); break;
    case 'get_hadith':    handleGetHadith(); break;
    case 'get_locations': handleGetLocations(); break;

    // --- Admin endpoints ---
    case 'login':         handleLogin(); break;
    case 'save_prayers':  requireAdmin(); handleSavePrayers(); break;
    case 'save_schedule': requireAdmin(); handleSaveSchedule(); break;
    case 'delete_schedule': requireAdmin(); handleDeleteSchedule(); break;
    case 'save_dua':      requireAdmin(); handleSaveDua(); break;
    case 'delete_dua':    requireAdmin(); handleDeleteDua(); break;
    case 'save_hadith':   requireAdmin(); handleSaveHadith(); break;
    case 'delete_hadith': requireAdmin(); handleDeleteHadith(); break;
    case 'save_location': requireAdmin(); handleSaveLocation(); break;
    case 'delete_location': requireAdmin(); handleDeleteLocation(); break;
    case 'change_password': requireAdmin(); handleChangePassword(); break;

    default:
        jsonResponse(['message' => 'রমযান বাড়ি API v1.0', 'status' => 'running']);
}

// ============================================================
// GET ALL (single call for viewer)
// ============================================================
function handleGetAll() {
    $db = getDB();
    
    $prayers = $db->query('SELECT prayer_key, prayer_name, prayer_time FROM prayers ORDER BY id')->fetchAll();
    $schedule = $db->query('SELECT * FROM schedule ORDER BY ramadan_day')->fetchAll();
    $duas = $db->query('SELECT * FROM duas ORDER BY sort_order, id')->fetchAll();
    $hadith = $db->query('SELECT * FROM hadith WHERE is_active = 1')->fetchAll();
    $locations = $db->query('SELECT * FROM locations ORDER BY is_default DESC, city')->fetchAll();
    
    // Format prayers as key=>value
    $prayerMap = [];
    foreach ($prayers as $p) {
        $prayerMap[$p['prayer_key']] = $p['prayer_time'];
    }
    
    jsonResponse([
        'prayers'   => $prayerMap,
        'schedule'  => $schedule,
        'duas'      => $duas,
        'hadith'    => $hadith,
        'locations' => $locations,
        'updated_at'=> date('Y-m-d H:i:s'),
    ]);
}

// ============================================================
// GET PRAYERS
// ============================================================
function handleGetPrayers() {
    $db = getDB();
    $rows = $db->query('SELECT prayer_key, prayer_name, prayer_time, updated_at FROM prayers ORDER BY id')->fetchAll();
    jsonResponse($rows);
}

// ============================================================
// SAVE PRAYERS (Admin)
// ============================================================
function handleSavePrayers() {
    $input = getInput();
    if (empty($input)) jsonError('No data provided');
    
    $db = getDB();
    $stmt = $db->prepare('UPDATE prayers SET prayer_time = ? WHERE prayer_key = ?');
    
    $updated = 0;
    foreach ($input as $key => $time) {
        if (!empty($key) && !empty($time)) {
            $stmt->execute([trim($time), trim($key)]);
            $updated += $stmt->rowCount();
        }
    }
    jsonResponse(['updated' => $updated, 'message' => 'নামাজের সময় আপডেট হয়েছে']);
}

// ============================================================
// GET SCHEDULE
// ============================================================
function handleGetSchedule() {
    $db = getDB();
    $rows = $db->query('SELECT * FROM schedule ORDER BY ramadan_day')->fetchAll();
    jsonResponse($rows);
}

// ============================================================
// SAVE SCHEDULE (Admin) - Insert or Update
// ============================================================
function handleSaveSchedule() {
    $input = getInput();
    $required = ['ramadan_day', 'day_name', 'hijri_date', 'sahri_time', 'fajr_time', 'iftar_time'];
    foreach ($required as $f) {
        if (empty($input[$f])) jsonError("Missing field: $f");
    }
    
    $db = getDB();
    
    if (!empty($input['id'])) {
        // Update existing
        $stmt = $db->prepare('UPDATE schedule SET ramadan_day=?, day_name=?, hijri_date=?, sahri_time=?, fajr_time=?, iftar_time=? WHERE id=?');
        $stmt->execute([$input['ramadan_day'], $input['day_name'], $input['hijri_date'], $input['sahri_time'], $input['fajr_time'], $input['iftar_time'], $input['id']]);
        jsonResponse(['id' => $input['id'], 'message' => 'সময়সূচী আপডেট হয়েছে']);
    } else {
        // Insert new
        $stmt = $db->prepare('INSERT INTO schedule (ramadan_day, day_name, hijri_date, sahri_time, fajr_time, iftar_time) VALUES (?,?,?,?,?,?) ON DUPLICATE KEY UPDATE day_name=VALUES(day_name), hijri_date=VALUES(hijri_date), sahri_time=VALUES(sahri_time), fajr_time=VALUES(fajr_time), iftar_time=VALUES(iftar_time)');
        $stmt->execute([$input['ramadan_day'], $input['day_name'], $input['hijri_date'], $input['sahri_time'], $input['fajr_time'], $input['iftar_time']]);
        jsonResponse(['id' => $db->lastInsertId(), 'message' => 'সময়সূচী যোগ হয়েছে']);
    }
}

// ============================================================
// DELETE SCHEDULE (Admin)
// ============================================================
function handleDeleteSchedule() {
    $input = getInput();
    if (empty($input['id'])) jsonError('ID required');
    $db = getDB();
    $db->prepare('DELETE FROM schedule WHERE id = ?')->execute([$input['id']]);
    jsonResponse(['message' => 'মুছে ফেলা হয়েছে']);
}

// ============================================================
// GET DUAS
// ============================================================
function handleGetDuas() {
    $db = getDB();
    $rows = $db->query('SELECT * FROM duas ORDER BY sort_order, id')->fetchAll();
    jsonResponse($rows);
}

// ============================================================
// SAVE DUA (Admin)
// ============================================================
function handleSaveDua() {
    $input = getInput();
    if (empty($input['title']) || empty($input['arabic_text']) || empty($input['bangla_translation'])) {
        jsonError('Required fields: title, arabic_text, bangla_translation');
    }
    
    $db = getDB();
    $hasAudio = isset($input['has_audio']) ? (int)$input['has_audio'] : 0;
    $sortOrder = isset($input['sort_order']) ? (int)$input['sort_order'] : 0;
    
    if (!empty($input['id'])) {
        $stmt = $db->prepare('UPDATE duas SET title=?, arabic_text=?, bangla_translation=?, has_audio=?, sort_order=? WHERE id=?');
        $stmt->execute([$input['title'], $input['arabic_text'], $input['bangla_translation'], $hasAudio, $sortOrder, $input['id']]);
        jsonResponse(['id' => $input['id'], 'message' => 'দোয়া আপডেট হয়েছে']);
    } else {
        $stmt = $db->prepare('INSERT INTO duas (title, arabic_text, bangla_translation, has_audio, sort_order) VALUES (?,?,?,?,?)');
        $stmt->execute([$input['title'], $input['arabic_text'], $input['bangla_translation'], $hasAudio, $sortOrder]);
        jsonResponse(['id' => $db->lastInsertId(), 'message' => 'দোয়া যোগ হয়েছে']);
    }
}

// ============================================================
// DELETE DUA (Admin)
// ============================================================
function handleDeleteDua() {
    $input = getInput();
    if (empty($input['id'])) jsonError('ID required');
    $db = getDB();
    $db->prepare('DELETE FROM duas WHERE id = ?')->execute([$input['id']]);
    jsonResponse(['message' => 'দোয়া মুছে ফেলা হয়েছে']);
}

// ============================================================
// GET HADITH
// ============================================================
function handleGetHadith() {
    $db = getDB();
    $all = isset($_GET['all']) && $_GET['all'] === '1';
    $sql = $all ? 'SELECT * FROM hadith ORDER BY id' : 'SELECT * FROM hadith WHERE is_active = 1 ORDER BY id';
    $rows = $db->query($sql)->fetchAll();
    jsonResponse($rows);
}

// ============================================================
// SAVE HADITH (Admin)
// ============================================================
function handleSaveHadith() {
    $input = getInput();
    if (empty($input['arabic_text']) || empty($input['bangla_translation']) || empty($input['source'])) {
        jsonError('Required fields: arabic_text, bangla_translation, source');
    }
    
    $db = getDB();
    $isActive = isset($input['is_active']) ? (int)$input['is_active'] : 1;
    
    if (!empty($input['id'])) {
        $stmt = $db->prepare('UPDATE hadith SET arabic_text=?, bangla_translation=?, source=?, is_active=? WHERE id=?');
        $stmt->execute([$input['arabic_text'], $input['bangla_translation'], $input['source'], $isActive, $input['id']]);
        jsonResponse(['id' => $input['id'], 'message' => 'হাদীস আপডেট হয়েছে']);
    } else {
        $stmt = $db->prepare('INSERT INTO hadith (arabic_text, bangla_translation, source, is_active) VALUES (?,?,?,?)');
        $stmt->execute([$input['arabic_text'], $input['bangla_translation'], $input['source'], $isActive]);
        jsonResponse(['id' => $db->lastInsertId(), 'message' => 'হাদীস যোগ হয়েছে']);
    }
}

// ============================================================
// DELETE HADITH (Admin)
// ============================================================
function handleDeleteHadith() {
    $input = getInput();
    if (empty($input['id'])) jsonError('ID required');
    $db = getDB();
    $db->prepare('DELETE FROM hadith WHERE id = ?')->execute([$input['id']]);
    jsonResponse(['message' => 'হাদীস মুছে ফেলা হয়েছে']);
}

// ============================================================
// GET LOCATIONS
// ============================================================
function handleGetLocations() {
    $db = getDB();
    $rows = $db->query('SELECT * FROM locations ORDER BY is_default DESC, city')->fetchAll();
    jsonResponse($rows);
}

// ============================================================
// SAVE LOCATION (Admin)
// ============================================================
function handleSaveLocation() {
    $input = getInput();
    if (empty($input['city']) || empty($input['country_code']) || !isset($input['latitude']) || !isset($input['longitude'])) {
        jsonError('Required fields: city, country_code, latitude, longitude');
    }
    
    $db = getDB();
    $isDefault = isset($input['is_default']) ? (int)$input['is_default'] : 0;
    $method = isset($input['calc_method']) ? (int)$input['calc_method'] : 3;
    
    if ($isDefault) {
        $db->exec('UPDATE locations SET is_default = 0');
    }
    
    if (!empty($input['id'])) {
        $stmt = $db->prepare('UPDATE locations SET city=?, country_code=?, latitude=?, longitude=?, calc_method=?, is_default=? WHERE id=?');
        $stmt->execute([$input['city'], strtoupper($input['country_code']), $input['latitude'], $input['longitude'], $method, $isDefault, $input['id']]);
        jsonResponse(['id' => $input['id'], 'message' => 'লোকেশন আপডেট হয়েছে']);
    } else {
        $stmt = $db->prepare('INSERT INTO locations (city, country_code, latitude, longitude, calc_method, is_default) VALUES (?,?,?,?,?,?)');
        $stmt->execute([$input['city'], strtoupper($input['country_code']), $input['latitude'], $input['longitude'], $method, $isDefault]);
        jsonResponse(['id' => $db->lastInsertId(), 'message' => 'লোকেশন যোগ হয়েছে']);
    }
}

// ============================================================
// DELETE LOCATION (Admin)
// ============================================================
function handleDeleteLocation() {
    $input = getInput();
    if (empty($input['id'])) jsonError('ID required');
    $db = getDB();
    $db->prepare('DELETE FROM locations WHERE id = ?')->execute([$input['id']]);
    jsonResponse(['message' => 'লোকেশন মুছে ফেলা হয়েছে']);
}

// ============================================================
// LOGIN (Admin)
// ============================================================
function handleLogin() {
    $input = getInput();
    if (empty($input['username']) || empty($input['password'])) {
        jsonError('Username and password required');
    }
    
    $db = getDB();
    $stmt = $db->prepare('SELECT * FROM admin_users WHERE username = ? LIMIT 1');
    $stmt->execute([$input['username']]);
    $user = $stmt->fetch();
    
    if (!$user || !password_verify($input['password'], $user['password_hash'])) {
        // Check legacy plain text passwords for initial setup
        if (!$user || $user['password_hash'] !== $input['password']) {
            // Special case: default password check
            if (!($user && $input['password'] === 'ramadan2026' && $user['username'] === 'admin')) {
                jsonError('Invalid credentials', 401);
            }
        }
    }
    
    // Generate token
    $ts = time();
    $hash = hash('sha256', $user['username'] . $ts . ADMIN_SESSION_KEY);
    $token = base64_encode($user['username'] . ':' . $ts . ':' . $hash);
    
    jsonResponse([
        'token'    => $token,
        'username' => $user['username'],
        'expires'  => $ts + 86400,
        'message'  => 'লগইন সফল হয়েছে'
    ]);
}

// ============================================================
// CHANGE PASSWORD (Admin)
// ============================================================
function handleChangePassword() {
    $input = getInput();
    if (empty($input['new_password'])) jsonError('New password required');
    if (strlen($input['new_password']) < 6) jsonError('Password must be at least 6 characters');
    
    $db = getDB();
    $hash = password_hash($input['new_password'], PASSWORD_BCRYPT);
    
    // Get current user from token
    $token = $_SERVER['HTTP_X_ADMIN_TOKEN'] ?? '';
    $decoded = base64_decode($token);
    list($user) = explode(':', $decoded, 2);
    
    if (!empty($input['new_username'])) {
        $stmt = $db->prepare('UPDATE admin_users SET username=?, password_hash=? WHERE username=?');
        $stmt->execute([$input['new_username'], $hash, $user]);
    } else {
        $stmt = $db->prepare('UPDATE admin_users SET password_hash=? WHERE username=?');
        $stmt->execute([$hash, $user]);
    }
    
    jsonResponse(['message' => 'পাসওয়ার্ড আপডেট হয়েছে']);
}
